<?php

/**
 * Functions
 *
 * @package     AutomatorWP\UserFeedback\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Options callback for select fields assigned to survey
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_userfeedback_options_cb_form($field)
{

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __('any survey', 'automatorwp-pro');
    $options = automatorwp_options_cb_none_option($field, $none_value, $none_label);

    if (!empty($value)) {
        if (!is_array($value)) {
            $value = array($value);
        }

        foreach ($value as $survey_id) {

            // Skip option none
            if ($survey_id === $none_value) {
                continue;
            }

            $options[$survey_id] = automatorwp_userfeedback_get_survey_title($survey_id);
        }
    }

    return $options;
}

/**
 * Options callback for select fields assigned to survey
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_userfeedback_survey_get_all_type_options_cb($field)
{
    
    $options = array(
        'long-text' => 'Long Text'
    );

    return $options;
}

/**
 * Get the survey title
 *
 * @since 1.0.0
 *
 * @param int $survey_id
 *
 * @return string|null
 */
function automatorwp_userfeedback_get_survey_title($survey_id)
{

    // Empty title if no ID provided
    if (absint($survey_id) === 0) {
        return '';
    }

    $surveys = array();
    $surveyClass = new UserFeedback_Survey();
    $all_object_surveys = $surveyClass->all();

    $all_array_surveys = json_decode(json_encode($all_object_surveys), true);

    foreach ($all_array_surveys as $survey) {
        if ($survey['status'] !== 'trash') {
            if (absint($survey_id) === absint($survey['id'])) {
                $survey_title = $survey['title'];
            }
        }
    }

    return $survey_title;
}

/**
 * Get the survey question
 *
 * @since 1.0.0
 *
 * @param int $survey_id
 *
 * @return string|null
 */
function automatorwp_userfeedback_get_survey_question($survey_id)
{

    // Empty title if no ID provided
    if (absint($survey_id) === 0) {
        return '';
    }

    $surveys = array();
    $surveyClass = new UserFeedback_Survey();
    $all_object_surveys = $surveyClass->all();

    $all_array_surveys = json_decode(json_encode($all_object_surveys), true);

    foreach ($all_array_surveys as $survey) {
        if ($survey['status'] !== 'trash') {
            if (absint($survey_id) === absint($survey['id'])) {
                $survey_question = $survey['questions'][0]['title'];
            }
        }
    }

    return $survey_question;
}

/**
 * Get all the responses
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_userfeedback_get_all_responses( $request ) {

    $array_responses = [];

    if( ! is_array( $request ) ) {
        return $array_responses;
    }

    foreach ($request as $response) {

        $array_responses[] = $response['value'];
    }

    return implode( ', ', $array_responses );

}

/**
 * Get all the responses
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_userfeedback_get_user_ip( $response_id ) {

    global $wpdb;

    $survey_response = $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}userfeedback_survey_responses WHERE id = %d", $response_id );

    $results = $wpdb->get_results( $survey_response );
    
    return $results[0]->user_ip;
}

/**
 * Get the surveys
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_userfeedback_get_surveys()
{

    $surveys = array();
    $surveyClass = new UserFeedback_Survey();
    $all_object_surveys = $surveyClass->all();

    $all_array_surveys = json_decode(json_encode($all_object_surveys), true);

    foreach ($all_array_surveys as $survey) {
        if ($survey['status'] !== 'trash') {
            $surveys[] = array(
                'id' => $survey['id'],
                'title'  => $survey['title'],
            );
        }
    }

    return $surveys;
}

function automatorwp_userfeedback_args_formatter($options){
    
    $return = array(
        'title' => $options['survey_title'],
        'status' => 'publish',
        'questions' => array(
            array(
                'id' => '',
                'type' => $options['survey_type'],
                'title' => $options['survey_question'],
                'config' => array(),
                'settings' => array(
                    'required' => '',
                    'randomize' => '',
                    'comment_box' => ''
                ),
                'logic' => array(
                    'enabled' => '',
                    'action' => 'show',
                    'conditions' => array()
                )
            )
        ),
        'settings' => array(
            'type' => 'message',
            'message' => 'Thanks for your feedback!',
            'redirect_to' => array(
                'id' => '',
                'label' => ''
            ),
            'enable_tracking' => '',
            'targeting' => array(
                'devices' => array(
                    'desktop',
                    'tablet',
                    'mobile'
                ),
                'pages' => 'all',
                'page_rules' => array()
            ),
            'behavior' => array(
                'timing' => 'immediate',
                'timing_delay' => 20,
                'display_length' => 'until_response',
                'run_time' => 'indefinitely',
                'expire_at' => array(
                    'month' => '',
                    'day' => ''
                ),
                'start_minimized' => ''
            ),
            'theming' => array(
                'color_scheme' => 'light',
                'widget_color' => '#ffffff',
                'text_color' => '#23282D',
                'button_color' => '#2D87F1'
            )
        ),
        'notifications' => array(
            'email' => array(
                'addresses' => '',
                'active' => 1,
                'logic' => array(
                    'enable' => '',
                    'conditions' => array(),
                    'action' => 'send'
                )
            )
        ),
        'impressions' => 0,
        'publish_at' => null
    );

    return $return;
    
}

/**
 * Custom tags replacements
 *
 * @since 1.0.0
 *
 * @param string    $parsed_content     Content parsed
 * @param array     $replacements       Automation replacements
 * @param int       $automation_id      The automation ID
 * @param int       $user_id            The user ID
 * @param string    $content            The content to parse
 *
 * @return string
 */
function automatorwp_userfeedback_parse_automation_tags($parsed_content, $replacements, $automation_id, $user_id, $content)
{

    $new_replacements = array();

    // Get automation triggers to pass their tags
    $triggers = automatorwp_get_automation_triggers($automation_id);

    foreach ($triggers as $trigger) {

        $trigger_args = automatorwp_get_trigger($trigger->type);

        // Skip if trigger is not from this integration
        if ($trigger_args['integration'] !== 'userfeedback') {
            continue;
        }

        // Get the last trigger log (where data for tags replacement will be get
        $log = automatorwp_get_user_last_completion($trigger->id, $user_id, 'trigger');

        if (!$log) {
            continue;
        }

        ct_setup_table('automatorwp_logs');
        $survey_fields = ct_get_object_meta($log->id, 'survey_fields', true);
        ct_reset_setup_table();

        // Skip if not survey fields
        if (!is_array($survey_fields)) {
            continue;
        }

        // Look for survey field tags
        preg_match_all("/\{t:" . $trigger->id . ":survey_field:\s*(.*?)\s*\}/", $parsed_content, $matches);

        if (is_array($matches) && isset($matches[1])) {

            foreach ($matches[1] as $field_name) {
                // Replace {t:ID:survey_field:NAME} by the field value
                if (isset($survey_fields[$field_name])) {
                    $new_replacements['{t:' . $trigger->id . ':survey_field:' . $field_name . '}'] = $survey_fields[$field_name];
                }
            }
        }

        // Look for survey field tags
        preg_match_all("/\{" . $trigger->id . ":survey_field:\s*(.*?)\s*\}/", $parsed_content, $matches);

        if (is_array($matches) && isset($matches[1])) {

            foreach ($matches[1] as $field_name) {
                // Replace {ID:survey_field:NAME} by the field value
                if (isset($survey_fields[$field_name])) {
                    $new_replacements['{' . $trigger->id . ':survey_field:' . $field_name . '}'] = $survey_fields[$field_name];
                }
            }
        }
    }

    if (count($new_replacements)) {

        $tags = array_keys($new_replacements);

        // Replace all tags by their replacements
        $parsed_content = str_replace($tags, $new_replacements, $parsed_content);
    }

    return $parsed_content;
}
add_filter('automatorwp_parse_automation_tags', 'automatorwp_userfeedback_parse_automation_tags', 10, 5);
